% ***********************************************************************
% *                          Simulation 1                               *    
% ***********************************************************************

%% =======Hodgkin-Huxley Space-Clamped Neuron Model Simulation===========
%
% Simulates neuron membrane potential using the Hodgkin-Huxley model with 
% voltage-gated Na⁺, K⁺, and leak channels. User-configurable parameters 
% control conductances, reversal potentials, and external current stimuli. 
% The model uses the Euler method to integrate gating variables and
% membrane voltage over time. Outputs plots of membrane voltage, 
% conductances, gating variables, and stimulus.

% Author: Oliver Foley
% Date: 29/04/2025

% Parts of code adapted from:
% File Name: NumHH.m
% Paper Title: Optimised Architectures and Implementations for Efficient Neuromorphic Hardware Design
% Author: Jonathan G-H-Cater
% Date: 08/04/2016

%% ========================= CONSTANTS =========================
% Define biophysical constants for ion channel conductances and reversal potentials
Cm = 1.0;            % Membrane capacitance (uF/cm^2)
gNa_max = 120;       % Maximum sodium conductance (mS/cm^2)
gK_max = 36;         % Maximum potassium conductance (mS/cm^2)
gL = 0.3;            % Leak conductance (mS/cm^2)
ENa = 56;            % Sodium reversal potential (mV)
EK = -77;            % Potassium reversal potential (mV)
EL = -68;            % Leak reversal potential (mV)

%% ========================= TIME =========================
dt = 0.1;               % Time step (ms)
tmax = 100;             % Total simulation time (ms)
T = 0:dt:tmax;          % Time vector (ms)

%% ========================= STIMULUS =========================
% Define stimulus parameters and convert to current density
I_stim = zeros(size(T));        % Initialise stimulation current (uA/cm^2)
I_ext = zeros(size(T));         % Initialise external current for plotting (nA)
axon_radius = 0.0001;           % Axon radius (cm)
axon_length1 = 1;               % Axon length (cm)
dx = axon_length1;              % For single compartment model, dx = full length
A = 2 * pi * axon_radius * dx;  % Surface area of the cylindrical segment (cm^2)

Amp1 = 5 / 1000;                % Stimulus 1 amplitude in mA
stim1 = Amp1 / A;               % Stimulus 1 current density (uA/cm^2)
Duration1 = 2;                  % Duration of stimulus 1 (ms)
Delay1 = 20;                    % Start delay for stimulus 1 (ms)

Amp2 = 10 / 1000;               % Stimulus 2 amplitude in mA
stim2 = Amp2 / A;               % Stimulus 2 current density (uA/cm^2)
Duration2 = 30;                 % Duration of stimulus 2 (ms)
Delay2 = 20;                    % Delay after stimulus 1 ends before starting stimulus 2 (ms)

stim_intervals = [Delay1, Delay1 + Duration1; ...
    Delay1 + Duration1 + Delay2, Delay1 + Duration1 + Delay2 + Duration2];
I_values = [5, 10];                     % External currents (nA) for visualisation
Stim_values = [stim1, stim2];          % Internal current densities (uA/cm^2)

% Apply stimulation during the specified intervals
for k = 1:size(stim_intervals, 1)
    I_ext(T >= stim_intervals(k, 1) & T <= stim_intervals(k, 2)) = I_values(k);
    I_stim(T >= stim_intervals(k, 1) & T <= stim_intervals(k, 2)) = Stim_values(k);
end

%% ========================= INITIAL CONDITIONS =========================
% Initialise membrane potential and gating variables
V = zeros(size(T)); V(1) = EL;         % Membrane potential (mV), start at resting
n = zeros(size(T)); m = zeros(size(T)); h = zeros(size(T));
n(1) = an(V(1)) / (an(V(1)) + bn(V(1)));  % Initial K+ activation
m(1) = am(V(1)) / (am(V(1)) + bm(V(1)));  % Initial Na+ activation
h(1) = ah(V(1)) / (ah(V(1)) + bh(V(1)));  % Initial Na+ inactivation

%% ========================= EULER SIMULATION =========================
% Loop through time using Euler integration to solve HH dynamics
for i = 1:length(T)-1
    I = I_stim(i);  % Injected current at time step i

    % Compute gating time constants
    tau_n = 1 / (an(V(i)) + bn(V(i)));
    tau_m = 1 / (am(V(i)) + bm(V(i)));
    tau_h = 1 / (ah(V(i)) + bh(V(i)));

    % Compute steady-state values for gating variables
    n_inf = an(V(i)) / (an(V(i)) + bn(V(i)));
    m_inf = am(V(i)) / (am(V(i)) + bm(V(i)));
    h_inf = ah(V(i)) / (ah(V(i)) + bh(V(i)));

    % Update gating variables using exponential Euler update
    n(i+1) = n_inf - (n_inf - n(i)) * exp(-dt / tau_n);
    m(i+1) = m_inf - (m_inf - m(i)) * exp(-dt / tau_m);
    h(i+1) = h_inf - (h_inf - h(i)) * exp(-dt / tau_h);

    % Calculate individual and total conductances
    gK = gK_max * n(i)^4;
    gNa = gNa_max * m(i)^3 * h(i);
    g_total = gK + gNa + gL;

    % Compute membrane time constant and steady-state voltage
    tau_V = Cm / g_total;
    V_inf = (gK * EK + gNa * ENa + gL * EL + I) / g_total;

    % Update membrane voltage
    V(i+1) = V_inf - (V_inf - V(i)) * exp(-dt / tau_V);
end

%% ========================= PLOTTING =========================
figure('Color', 'w', 'Position', [100 100 700 800]);

% Plot 1: Stimulus current
subplot(4,1,1)
plot(T, I_ext, 'k', 'LineWidth', 1.5);
ylabel('Current (nA)', 'FontSize', 10); 
title('External Stimulation Current', 'FontSize', 11, 'FontWeight', 'bold');
grid on; xlim([0, max(T)]);
xlabel('Time (ms)', 'FontSize', 10);
ylim([0, max(I_ext)+1]);

% Plot 2: Membrane potential
subplot(4,1,2)
plot(T, V, 'b', 'LineWidth', 2);
ylabel('Membrane Voltage (mV)', 'FontSize', 10); 
title('Membrane Voltage Response', 'FontSize', 11, 'FontWeight', 'bold');
grid on;
xlabel('Time (ms)', 'FontSize', 10);

% Plot 3: Conductances
subplot(4,1,3)
plot(T, gK_max * (n.^4), 'Color', '#F80', 'LineWidth', 1.5); hold on;
plot(T, gNa_max * (m.^3) .* h, 'g-', 'LineWidth', 1.5);
xlabel('Time (ms)', 'FontSize', 10);
ylabel('Conductance (mS/cm²)', 'FontSize', 10);
title('Sodium and Potassium Conductances', 'FontSize', 11, 'FontWeight', 'bold');
legend('K⁺ Conductance', 'Na⁺ Conductance', 'Location', 'best');
grid on; hold off;

% Plot 4: Gating variables
subplot(4,1,4)
plot(T, n, 'g-', 'LineWidth', 1.5); hold on;
plot(T, m, 'r-', 'LineWidth', 1.5);
plot(T, h, 'b-', 'LineWidth', 1.5);
xlabel('Time (ms)', 'FontSize', 10); 
ylabel('Gating Variables', 'FontSize', 10);
title('Gating Variable Dynamics', 'FontSize', 11, 'FontWeight', 'bold');
legend('n (K⁺ activation)', 'm (Na⁺ activation)', 'h (Na⁺ inactivation)', 'Location', 'best');
grid on; hold off;

% Overall title for all plots
sgtitle('Simulation 1: Action Potentials – Single Compartment Hodgkin-Huxley Neuron Model ', ...
    'FontSize', 13, 'FontWeight', 'bold');

%% ========================= GATING FUNCTIONS =========================
function val = an(V)
    val = 0.01 * (10 - (V + 71)) ./ (exp(1 - (V + 71) / 10) - 1); % alpha_n
end

function val = bn(V)
    val = 0.125 * exp(-(V + 71) / 80);                             % beta_n
end

function val = am(V)
    val = 0.1 * (25 - (V + 71)) ./ (exp(2.5 - (V + 71) / 10) - 1); % alpha_m
end

function val = bm(V)
    val = 4 * exp(-(V + 71) / 18);                                 % beta_m
end

function val = ah(V)
    val = 0.07 * exp(-(V + 71) / 20);                              % alpha_h
end

function val = bh(V)
    val = 1 ./ (exp(3 - (V + 71) / 10) + 1);                       % beta_h
end
